// SPDX-License-Identifier: MIT
pragma solidity ^0.8.24;

import {Ownable} from "openzeppelin-contracts/contracts/access/Ownable.sol";

/**
 * @title TicketRegistry
 * @notice Registry for hospital maintenance tickets. Only the backend relayer
 *         (deployer/owner) can create and assign tickets. Status can be updated
 *         by owner or current assignee. No patient PII goes on-chain — only a hash.
 */
contract TicketRegistry is Ownable {
    /// @notice Lifecycle states for a ticket
    enum Status {
        Open,         // 0 - newly created
        InProgress,   // 1
        Resolved,     // 2
        Closed,       // 3
        Canceled      // 4
    }

    /// @notice Ticket record kept on-chain (no PII)
    struct Ticket {
        bytes32 hash;        // keccak256 of canonical JSON (server-side)
        uint8 severity;      // 0..n (app-defined)
        string department;   // e.g., "Facilities", "Electrical"
        address creator;     // msg.sender at creation (relayer/owner)
        address assignee;    // maintenance user/wallet if any
        Status status;       // lifecycle status
        uint64 createdAt;    // block timestamp (cast)
        uint64 updatedAt;    // last mutation time
    }

    /// @notice auto-incrementing id
    uint256 public nextId = 1;

    /// @dev id => Ticket
    mapping(uint256 => Ticket) private _tickets;

    /// @notice Emitted on create
    event TicketCreated(
        uint256 indexed id,
        bytes32 indexed hash,
        uint8 severity,
        string department,
        address indexed creator
    );

    /// @notice Emitted on assignment change
    event TicketAssigned(uint256 indexed id, address indexed assignee);

    /// @notice Emitted on status change
    event TicketStatusChanged(uint256 indexed id, Status newStatus);

    constructor(address initialOwner) Ownable(initialOwner) {}

    /**
     * @notice Create a new ticket (only backend relayer/owner).
     * @param jsonHash  keccak256 hash of canonical JSON computed server-side
     * @param severity  app-defined severity (0..n)
     * @param department free-text small string like "Facilities"
     */
    function createTicket(
        bytes32 jsonHash,
        uint8 severity,
        string calldata department
    ) external onlyOwner returns (uint256 id) {
        require(jsonHash != bytes32(0), "hash required");
        id = nextId++;
        Ticket storage t = _tickets[id];
        t.hash = jsonHash;
        t.severity = severity;
        t.department = department;
        t.creator = _msgSender();
        t.status = Status.Open;
        t.createdAt = uint64(block.timestamp);
        t.updatedAt = uint64(block.timestamp);

        emit TicketCreated(id, jsonHash, severity, department, t.creator);
    }

    /**
     * @notice Assign a ticket to an address (only owner).
     */
    function assignTicket(uint256 id, address assignee) external onlyOwner {
        Ticket storage t = _requireTicket(id);
        t.assignee = assignee;
        t.updatedAt = uint64(block.timestamp);
        emit TicketAssigned(id, assignee);
    }

    /**
     * @notice Update status. Owner or current assignee can update.
     */
    function updateStatus(uint256 id, Status newStatus) external {
        Ticket storage t = _requireTicket(id);
        require(
            _msgSender() == owner() || _msgSender() == t.assignee,
            "not authorized"
        );
        t.status = newStatus;
        t.updatedAt = uint64(block.timestamp);
        emit TicketStatusChanged(id, newStatus);
    }

    /**
     * @notice Read a ticket.
     */
    function getTicket(uint256 id)
        external
        view
        returns (
            bytes32 hash_,
            uint8 severity_,
            string memory department_,
            address creator_,
            address assignee_,
            Status status_,
            uint64 createdAt_,
            uint64 updatedAt_
        )
    {
        Ticket storage t = _requireTicket(id);
        return (
            t.hash,
            t.severity,
            t.department,
            t.creator,
            t.assignee,
            t.status,
            t.createdAt,
            t.updatedAt
        );
    }

    /**
     * @notice Returns true if a ticket exists.
     */
    function exists(uint256 id) external view returns (bool) {
        return _exists(id);
    }

    /**
     * @notice Count of created tickets (ids go from 1..nextId-1).
     */
    function totalTickets() external view returns (uint256) {
        return nextId - 1;
    }

    // ----- internal helpers -----

    function _exists(uint256 id) internal view returns (bool) {
        return id > 0 && id < nextId && _tickets[id].createdAt != 0;
    }

    function _requireTicket(uint256 id) internal view returns (Ticket storage) {
        require(_exists(id), "ticket not found");
        return _tickets[id];
    }
}
